﻿#include "precompiled.h"
#include "common.h"
#include "RTCore.h"

#include "AppWindow.h"
#include "RTRenderer.h"
#include "RTSimulation.h"
#include "ResourceManager.h"
#include "BasicTimer.h"
#include "FpsTimer.h"
#include "Camera.h"
#include "Scene.h"
#include "SceneImporter.h"
#include "LensImporter.h"
#include "Entity.h"
#include "Transform.h"
#include "Camera.h"

#define PRINT_FPS

namespace RTCam {

RTCore::RTCore(void) :
	m_initialized(false),
	m_window(nullptr),
	m_scene(nullptr),
	m_resources(new ResourceManager()),
	m_updateTimer(new BasicTimer()),
	m_renderTimer(new BasicTimer()),
	m_updateFpsTimer(new FpsTimer()),
	m_renderFpsTimer(new FpsTimer()),
	m_renderer(new RTRenderer()),
	m_simulation(new RTSimulation()),
	m_sceneImporter(new SceneImporter())
{
}


RTCore::~RTCore(void)
{
}

HRESULT RTCore::Initialize(_In_ HINSTANCE hInstance, int nCmdShow)
{
	// Create and initialize the window.
	m_window = unique_ptr<AppWindow>(new AppWindow());
	if(FAILED(m_window->InitWindow(hInstance, nCmdShow))) {
		return E_FAIL;
	}

	// Initialize the renderer.
	m_renderer->Initialize(m_window);

	// Initialize the resource manager.
	m_resources->Initialize(m_renderer->m_d3dDevice.Get());

	// Initialize the scene importer.
	m_sceneImporter->Initialize(m_resources);

	// Initialize the default empty scene.
	//m_scene->Initialize(m_resources);
	
	// Initialize a specific scene.
	//m_scene = m_sceneImporter->InitCubeGridScene();
	//m_scene = m_sceneImporter->InitCubeTestScene();
	//m_scene = m_sceneImporter->InitHierarchyTestScene();
	m_scene = m_sceneImporter->InitLampsTestScene();
	//m_scene = m_sceneImporter->InitDepthTestScene();
	


	m_initialized = true;

	return S_OK;
}

void RTCore::FixedUpdate()
{
	// TODO: Make physics update calls occur at fixed time steps
	m_updateTimer->Update();
	float curTime = m_updateTimer->GetTotal();
	float delta = m_updateTimer->GetDelta();

	m_updateFpsTimer->Update(delta);
	string fpsText = StringFormat("Avg Update fps: %2.0f", m_updateFpsTimer->GetAvgFps());
	m_renderer->QueueUIText(fpsText.c_str(), 0, 0);

	m_simulation->FixedUpdate(m_scene, curTime, delta);
}

void RTCore::Render()
{
	m_renderTimer->Update();
	float curTime = m_renderTimer->GetTotal();
	float delta = m_renderTimer->GetDelta();

	m_renderFpsTimer->Update(delta);
	string fpsText = StringFormat("Avg Render fps: %2.0f", m_renderFpsTimer->GetAvgFps());
	m_renderer->QueueUIText(fpsText.c_str(), 0, Direct3DBase::RecommendedLineSpacing);

	// Render if window isn't minimized
	if(!IsIconic(m_window->GetHWND())) {
		m_simulation->RenderUpdate(m_scene, curTime, delta);

		m_renderer->RenderUpdate(m_scene, curTime, delta);
		m_renderer->Render(m_scene);
		m_renderer->Present();
	} else {
		// HACK: Sleep for 1/60 of a second (16.66... ms)
		Sleep(17);
		// END HACK
	}
}

void RTCore::OnResize()
{
	if(m_initialized) {
		m_window->OnResize();
		m_renderer->UpdateForWindowSizeChange();
	}
}

void RTCore::OnKeyUp( UINT keyCode )
{
	shared_ptr<Camera> camera = m_scene->m_activeCam.lock();
	if(camera == nullptr) {
		return;
	}

	switch(keyCode) {
	// Toggle the UI
	case VK_RETURN:
		m_renderer->m_drawUI = !m_renderer->m_drawUI;
		break;
	// Scene loading
	case 'O':
		{
			auto loadedScene = m_sceneImporter->OpenScene(m_renderer->m_d3dDevice.Get());
			if(loadedScene != nullptr) {
				m_scene = loadedScene;
			}
		}
		break;
	// Lens loading
	case 'L':
		{
			auto curCamera = m_scene->m_activeCam.lock();
			bool success = m_lensImporter->ImportLens(m_renderer->m_d3dDevice.Get(), curCamera.get());
			//if(!success) {
			//	// Reset the scene camera to a default camera
			//	m_scene->RemoveEntity(curCamera->m_entity.lock());
			//	curCamera = nullptr;
			//	m_scene->AddCamera();
			//}
		}
		break;
	// Camera movements
	case 'W':
		m_simulation->MoveCameraForward(camera);
		break;
	case 'S':
		m_simulation->MoveCameraBack(camera);
		break;
	case 'A':
		m_simulation->MoveCameraLeft(camera);
		break;
	case 'D':
		m_simulation->MoveCameraRight(camera);
		break;
	case 'Q':
		m_simulation->TurnCameraLeft(camera);
		break;
	case 'E':
		m_simulation->TurnCameraRight(camera);
		break;
	case 'R':
		m_simulation->MoveCameraUp(camera);
		break;
	case 'F':
		m_simulation->MoveCameraDown(camera);
		break;
	// Spacebar
	case VK_SPACE:
		m_renderer->ToggleLongExposureRendering();
		break;
	// Plus and minus
	case VK_OEM_MINUS:
		camera->DecreaseShutterSpeed();
		break;
	case VK_OEM_PLUS:
		camera->IncreaseShutterSpeed();
		break;
	// [ and ]
	case VK_OEM_4:
		camera->DecreaseFilmSpeed();
		break;
	case VK_OEM_6:
		camera->IncreaseFilmSpeed();
		break;
	// ; and '
	case VK_OEM_1:
		camera->DecreaseLensFocalLength();
		break;
	case VK_OEM_7:
		camera->IncreaseLensFocalLength();
		break;
	// Arrow keys
	case VK_LEFT:
		camera->IncreaseFNumber();
		break;
	case VK_RIGHT:
		camera->DecreaseFNumber();
		break;
	case VK_UP:
		camera->MoveFocusForward();
		break;
	case VK_DOWN:
		camera->MoveFocusBack();
		break;
	// Standard number keys (not on numpad)
	case '0':
		m_renderer->m_debugRenderMode = 0;
		break;
	case '1':
		m_renderer->m_debugRenderMode = 1;
		break;
	case '2':
		m_renderer->m_debugRenderMode = 2;
		break;
	case '3':
		m_renderer->m_debugRenderMode = 3;
		break;
	case '4':
		m_renderer->m_debugRenderMode = 4;
		break;
	case '5':
		m_renderer->m_debugRenderMode = 5;
		break;
	case '6':
		m_renderer->m_debugRenderMode = 6;
		break;
	case '7':
		m_renderer->m_debugRenderMode = 7;
		break;
	case '8':
		m_renderer->m_debugRenderMode = 8;
		break;
	case '9':
		m_renderer->m_debugRenderMode = 9;
		break;
	}
}

} // end namespace